breeds [ 
  plane 
  listener 
  wave-components 
]

wave-components-own [ 
  amplitude 
  wave-id ;; the wave-id identifies which wave this  
          ;; component is a part of
]

listener-own [
  wave-ids-heard ;; which wave-ids the listener just heard
                 ;; computed to avoid double-counting
]

globals [ 
  speed-of-sound  ;; constant
  next-wave-id ;; counters
  ticks 
  wave-interval ;; how many ticks between each wave?
  initial-wave-amplitude
]

to setup 
  ca
  set-default-shape wave-components "wave particle"
  set-default-shape plane "airplane"
  set-default-shape listener "person"
  
  set speed-of-sound 757
  set initial-wave-amplitude 20
  set wave-interval 3
  
  ;; initialize counters
  set ticks 0
  set next-wave-id 0
  
  ;; create the airplane
  create-custom-plane 1 [ 
    set heading 90 
    set ycor 3 - screen-edge-y
    set xcor 14 - screen-edge-x
    set size 4 
    set color white
  ]
  
  ;; create the listener
  create-custom-listener 1 [
    set ycor 1 - screen-edge-y
    set size 3
    set color blue
  ]
  
  ;; draw the ground
  ask patches with [ pycor = -1 * screen-edge-y ] 
    [ set pcolor grey ]
end

to go 
  ask plane [ fd 1 * plane-speed / speed-of-sound ]      ;; move the plane
  if ticks mod wave-interval = 0 [ ask plane [ emit-wave ] ] ;; emit the sound wave
  
  ;; move waves
  ask wave-components [ 
    fd 1
    if pcolor = grey [ die ] ;; the ground absorbes sound
    set amplitude amplitude - 1
    set color scale-color yellow amplitude 0 initial-wave-amplitude
    if amplitude < 1 [ die ]
  ]
  
  ;; listen and plot
  ask listener [ 
    let amp amplitude-here wave-ids-heard
    plotxy ticks amp
    plotxy ticks + 1 0
    set ticks ticks + 2
    set wave-ids-heard 
      remove-duplicates 
        values-from wave-components-here [ wave-id ]
  ]
  
  ;; draw 
  ifelse show-amplitudes? [
    ;; hide the wave and show total amplitude on each patch
    ask wave-components [ ht ]
    ask patches with [ pcolor != grey ] [ 
      let amp amplitude-here []
      ifelse amp > 0 
        [ set plabel amp ] 
        [ set plabel no-label ]
      set pcolor scale-color green amp 0 60
      set plabel-color black
    ]
  ] [
    ;; show the wave and paint patches black
    ask wave-components [ st ]
     ask patches with [ pcolor != grey ] [ 
       set pcolor black 
       set plabel no-label
    ]
  ]
end


;; patch procedure
;; counts the total amplitude of the waves on this patch, 
;; making sure not to count two components of the same wave.
to-report amplitude-here [ids-to-exclude]
  let total-amplitude 0
  let components wave-components-here
  if count components > 0 [ 
    ;; get list of the wave-ids with components on this patch
    let wave-ids-here remove-duplicates values-from components [ wave-id ]
    foreach ids-to-exclude [ set wave-ids-here remove ? wave-ids-here ]

    ;; for each wave id, sum the maximum amplitude here
    foreach wave-ids-here [ set total-amplitude total-amplitude + 
        amplitude-of max-one-of components with [ wave-id = ? ] 
          [ amplitude ]
    ]
  ]
  report total-amplitude
end

;; plane procedure
to emit-wave
  let j 0
  let num-wave-components 90 ;; number of components in each wave
  hatch-wave-components 360 [ 
    set color yellow 
    set size 1
    set j j + 1 
    set amplitude initial-wave-amplitude 
    set wave-id next-wave-id
    set heading j * ( 360.0 / num-wave-components )
    if show-amplitudes? [ ht ]
  ]
  set next-wave-id next-wave-id + 1
end

;; reports the plane speed in Mach, or 
;; number of times the speed of sound
to-report mach
  report plane-speed / speed-of-sound
end


; This model was created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; The project gratefully acknowledges the support of the
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2004 by Uri Wilensky.  Updated 2004.  All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2004).  NetLogo Doppler model.
; http://ccl.northwestern.edu/netlogo/models/Doppler.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2004 Uri Wilensky.  All rights reserved.
; See http://ccl.northwestern.edu/netlogo/models/Doppler
; for terms of use.
@#$#@#$#@
GRAPHICS-WINDOW
253
21
873
262
30
10
10.0
1
8
1
1
1
0

CC-WINDOW
5
431
882
526
Command Center

BUTTON
19
18
85
51
NIL
setup
NIL
1
T
OBSERVER
T
NIL

BUTTON
95
18
158
51
NIL
go
T
1
T
OBSERVER
T
NIL

PLOT
253
265
873
417
Signal
Time
Amplitude
0.0
10.0
0.0
10.0
true
false

MONITOR
169
19
226
68
Time
ticks
3
1

SLIDER
18
80
225
113
plane-speed
plane-speed
0
1514
0
1
1
miles per hour

MONITOR
18
164
75
213
NIL
Mach
3
1

BUTTON
18
123
225
156
NIL
set plane-speed speed-of-sound\n
NIL
1
T
OBSERVER
T
NIL

TEXTBOX
93
166
243
194
Mach 1 = 757 mph = speed-of-sound

SWITCH
15
337
185
370
show-amplitudes?
show-amplitudes?
1
1
-1000

@#$#@#$#@
WHAT IS IT?
-----------
This model demonstrates the Doppler effect, the apparent change in the frequency of a wave emitted by a source moving relative to an observer.

When the source of a wave moves towards you, the perceived frequency of the wave increases; when the source moves away from you, the perceived frequency decreases. This phenomena can be observed when a car passes you while the driver honks his horn. The pitch of the sound you hear is higher as the car approaches you and lower when the car is moving away. 

In this model, a plane flies above an observer. Yellow circles represent the peaks of sound waves emitted by the plane. 


HOW TO USE IT
--------------  
Press the SETUP button to create a plane and a person. Press GO to start the plane moving. Adjust the PLANE-SPEED slider to control the speed of the plane. The SHOW-AMPLITUDES? switch lets you see the strength of the sound wave on each patch of air.


THINGS TO NOTICE
----------------
Set the speed to zero. When the plane is not moving, the wavelength (the distance between the peaks of each wave) is the same on both sides of the plane. As you increase the speed of the plane, the waves bunch together in front of the plane and spread apart behind the plane. So when the plane is moving towards the person, the wavelength is shorter, so the perceived frequency of the sound is higher. When the plane is moving away from the person, the wavelength is longer, so the perceived frequency of the sound is lower.

When the plane is travelling at the speed of sound (Mach 1, approximately 757mph), notice how all the sound waves overlap at one point. At this point of intersection, the constructive interference of the wave peaks creates a loud bang called a SONIC BOOM.    

THINGS TO TRY
-------------

Set the plane speed to the speed of sound, 757 miles per hour (Mach 1). Notice that the peaks of the sound waves in front of the plane bunch up completely. Look at the SIGNAL plot when the plane passes the person at the speed of sound? When happens to the perceived amplitude of the sound heard by the person? This phenomena results from a shock wave -- the constructive interference of a large number of wave peaks -- and creates a very loud sound called a sonic boom. 

Turn on the SHOW-AMPLITUDES? switch and adjust the plane speed to watch the constructive interference in action as the plane speed approaches Mach 1.

Increase the plane speed beyond the speed of sound. What happens to the shape of the shock wave? What does the person hear?


EXTENDING THE MODEL
--------------------
Add a second plot that displays the relative frequency heard by the person. Improve the first plot to interpolate the signal data and display the amplitude between signal peaks.

In this model, only the plane is in motion. Add controls to move the person as well.

Use the NetLogo extensions API to write a Java extension that plays a sound at a given amplitude and frequency. Have the person generate the sound he hears so you can listen to the Doppler effect in action.
       

CREDITS AND REFERENCES
----------------------
To refer to this model in academic publications, please use: Wilensky, U. (2004).  NetLogo Doppler model. http://ccl.northwestern.edu/netlogo/models/Doppler. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2004 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/Doppler for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
false
0
Polygon -7566196 true true 0 90 45 135 255 135 300 195 0 195

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wave particle
true
0
Rectangle -7566196 true true 0 120 300 180

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
